<?php

include '../include/conn.php';

header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json");

// Step 1: Get and validate input
$raw = file_get_contents('php://input');
$input = json_decode($raw, true);

// Fallback to $_POST (handles form-data / x-www-form-urlencoded)
if (!is_array($input) || empty($input)) {
    $input = $_POST;
}

// Accept 'subscription_key' (instead of 'key')
if (!isset($input['subscription_key']) || empty(trim($input['subscription_key']))) {
    // Extension expects numeric success/data — return 0s for failure
    echo json_encode([
        "success" => 0,
        "data"    => 0,
        "message" => "Subscription key is missing."
    ]);
    exit;
}

$key = trim($input['subscription_key']);

// DB connection check
if ($conn->connect_error) {
    echo json_encode([
        "success" => 0,
        "data"    => 0,
        "message" => "Database connection failed: " . $conn->connect_error
    ]);
    exit;
}

// Step 2: Fetch license key
$stmt = $conn->prepare("SELECT id, license_key, is_active, expire_at, active_at FROM license_keys WHERE license_key = ? LIMIT 1");
$stmt->bind_param("s", $key);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows > 0) {
    $row = $result->fetch_assoc();
    $id = $row['id'];

    $updates = [];
    $params = [];

    // Step 3: Check and update expire_at if NULL
    if (is_null($row['expire_at'])) {
        $expire_at = date("Y-m-d H:i:s", strtotime("+365 days"));
        $updates[] = "expire_at = ?";
        $params[] = $expire_at;
    }

    // Step 4: Check and update active_at if NULL
    if (is_null($row['active_at'])) {
        $active_at = date("Y-m-d H:i:s");
        $updates[] = "active_at = ?";
        $params[] = $active_at;
    }

    // Step 5: If any updates needed, perform UPDATE query
    if (!empty($updates)) {
        $query = "UPDATE license_keys SET " . implode(', ', $updates) . " WHERE id = ?";
        $params[] = $id;

        $update_stmt = $conn->prepare($query);

        // Create dynamic bind types string
        $types = str_repeat('s', count($params) - 1) . 'i';
        $update_stmt->bind_param($types, ...$params);
        $update_stmt->execute();
        $update_stmt->close();

        // Also refresh $row from DB so that subsequent checks use updated values
        $stmt->execute();
        $result = $stmt->get_result();
        $row = $result->fetch_assoc();
    }

    // Step 6: Re-validate status (use updated $row)
    if (!$row['is_active']) {
        echo json_encode([
            "success" => 0,
            "data"    => 0,
            "message" => "License key is deactivated."
        ]);
    } elseif (!empty($row['expire_at']) && strtotime($row['expire_at']) < time()) {
        echo json_encode([
            "success" => 0,
            "data"    => 0,
            "message" => "License key has expired."
        ]);
    } else {
        // success — extension expects success:1 and data:1
        echo json_encode([
            "success" => 1,
            "data"    => 1,
            "message" => "License key is valid."
        ]);
    }

} else {
    echo json_encode([
        "success" => 0,
        "data"    => 0,
        "message" => "License key not found."
    ]);
}

$stmt->close();
$conn->close();
